# Docker Setup Guide for UNIOSUN Payment Backend

This guide will help you set up and deploy the UNIOSUN Payment Backend on a VPS using Docker.

## Prerequisites

- Docker Engine (version 20.10 or higher)
- Docker Compose (version 2.0 or higher)
- At least 2GB of available RAM
- At least 10GB of available disk space

## Quick Start

1. **Create a `.env` file** in the project root with the following variables:

```env
# Server Configuration
PORT=5000
NODE_ENV=production

# Database Configuration
DB_HOST=db
DB_USER=uniosunpay
DB_PASSWORD=uniosunpay123
DB_NAME=uniosun_payment
DB_ROOT_PASSWORD=rootpassword
DB_PORT=3306

# JWT Configuration
JWT_SECRET=your-super-secret-jwt-key-change-this-in-production
JWT_EXPIRES_IN=7d

# Remita Payment Gateway Configuration
REMITA_MERCHANT_ID=your-remita-merchant-id
REMITA_API_KEY=your-remita-api-key
REMITA_PUBLIC_KEY=your-remita-public-key
REMITA_ENV=demo
# REMITA_ENV options: 'demo' or 'production'

# Rate Limiting Configuration
RATE_LIMIT_WINDOW_MS=900000
RATE_LIMIT_MAX_REQUESTS=100
```

2. **Build and start the containers:**

```bash
docker-compose up -d
```

3. **Check the logs:**

```bash
docker-compose logs -f app
```

4. **Verify the application is running:**

```bash
curl http://localhost:5000/health
```

## Environment Variables

### Required Variables

- `JWT_SECRET`: Secret key for JWT token signing (use a strong random string)
- `REMITA_MERCHANT_ID`: Your Remita merchant ID
- `REMITA_API_KEY`: Your Remita API key
- `REMITA_PUBLIC_KEY`: Your Remita public key

### Optional Variables (with defaults)

- `PORT`: Application port (default: 5000)
- `NODE_ENV`: Environment mode (default: production)
- `DB_USER`: Database user (default: uniosunpay)
- `DB_PASSWORD`: Database password (default: uniosunpay123)
- `DB_NAME`: Database name (default: uniosun_payment)
- `DB_ROOT_PASSWORD`: MySQL root password (default: rootpassword)
- `DB_PORT`: MySQL port (default: 3306)
- `JWT_EXPIRES_IN`: JWT token expiration (default: 7d)
- `REMITA_ENV`: Remita environment - 'demo' or 'production' (default: demo)
- `RATE_LIMIT_WINDOW_MS`: Rate limit window in milliseconds (default: 900000 = 15 minutes)
- `RATE_LIMIT_MAX_REQUESTS`: Max requests per IP per window (default: 100)

## Docker Commands

### Start services
```bash
docker-compose up -d
```

### Stop services
```bash
docker-compose down
```

### View logs
```bash
# All services
docker-compose logs -f

# Application only
docker-compose logs -f app

# Database only
docker-compose logs -f db
```

### Restart services
```bash
docker-compose restart
```

### Rebuild and restart
```bash
docker-compose up -d --build
```

### Access database
```bash
docker-compose exec db mysql -u uniosunpay -p uniosun_payment
```

### Access application container
```bash
docker-compose exec app sh
```

## Database Initialization

If you have SQL initialization scripts, place them in the `mysql-init/` directory. Files in this directory will be executed automatically when the database container starts for the first time.

Example:
```bash
mkdir -p mysql-init
# Copy your database.sql or schema files to mysql-init/
```

## Data Persistence

- **Database data**: Stored in Docker volume `mysql_data` (persists even if containers are removed)
- **Uploads**: Stored in `./uploads` directory (mounted as volume)

## Production Deployment Checklist

1. ✅ Change all default passwords in `.env`
2. ✅ Generate a strong `JWT_SECRET` (use: `openssl rand -base64 32`)
3. ✅ Set `NODE_ENV=production`
4. ✅ Update `REMITA_ENV=production` with production credentials
5. ✅ Configure firewall to only allow necessary ports
6. ✅ Set up SSL/TLS (use reverse proxy like Nginx)
7. ✅ Configure backups for MySQL volume
8. ✅ Set up log rotation
9. ✅ Configure monitoring and alerts

## Reverse Proxy Setup (Nginx Example)

For production, it's recommended to use Nginx as a reverse proxy:

```nginx
server {
    listen 80;
    server_name your-domain.com;

    location / {
        proxy_pass http://localhost:5000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
    }
}
```

## Troubleshooting

### Database connection errors
- Ensure the database container is healthy: `docker-compose ps`
- Check database logs: `docker-compose logs db`
- Verify environment variables match in both services

### Application won't start
- Check application logs: `docker-compose logs app`
- Verify all required environment variables are set
- Ensure port 5000 is not already in use

### Permission errors with uploads
- Ensure `uploads/` directory exists and has proper permissions
- Run: `chmod -R 755 uploads/`

## Backup and Restore

### Backup database
```bash
docker-compose exec db mysqldump -u uniosunpay -p uniosun_payment > backup.sql
```

### Restore database
```bash
docker-compose exec -T db mysql -u uniosunpay -p uniosun_payment < backup.sql
```

## Security Notes

- Never commit `.env` file to version control
- Use strong passwords for database
- Keep Docker and images updated
- Regularly review and update dependencies
- Use secrets management in production (Docker secrets, AWS Secrets Manager, etc.)

